# Quick Start

# Compile a Job with the Burst compiler

Burst is primarily designed to work efficiently with the Job system. 

You can start using the Burst compiler in your code by simply decorating a Job struct with the attribute `[BurstCompile]`

```c#
using Unity.Burst;
using Unity.Collections;
using Unity.Jobs;
using UnityEngine;

public class MyBurst2Behavior : MonoBehaviour
{
    void Start()
    {
        var input = new NativeArray<float>(10, Allocator.Persistent);
        var output = new NativeArray<float>(1, Allocator.Persistent);
        for (int i = 0; i < input.Length; i++)
            input[i] = 1.0f * i;

        var job = new MyJob
        {
            Input = input,
            Output = output
        };
        job.Schedule().Complete();

        Debug.Log("The result of the sum is: " + output[0]);
        input.Dispose();
        output.Dispose();
    }

    // Using BurstCompile to compile a Job with Burst
    // Set CompileSynchronously to true to make sure that the method will not be compiled asynchronously
    // but on the first schedule
    [BurstCompile(CompileSynchronously = true)]
    private struct MyJob : IJob
    {
        [ReadOnly]
        public NativeArray<float> Input;

        [WriteOnly]
        public NativeArray<float> Output;

        public void Execute()
        {
            float result = 0.0f;
            for (int i = 0; i < Input.Length; i++)
            {
                result += Input[i];
            }
            Output[0] = result;
        }
    }
}
```

By default (only within the Editor - See [AOT vs JIT](#just-in-time-jit-vs-ahead-of-time-aot-compilation)), Burst JIT compiles jobs asynchronously, but the example above uses the option `CompileSynchronously = true` to make sure that the method is compiled on the first schedule. In general, you should use asynchronous compilation. See [`[BurstCompile]` options](AdvancedUsages.md#synchronous-compilation)

<a name="jobs-burst-menu"></a>
# Jobs/Burst Menu

The Burst package adds a few menu entries to the Jobs menu for controlling Burst behavior:

![Burst menu entries](../images/burst-menu.png)

- **Enable Compilation**: When checked, Burst compiles Jobs and Burst custom delegates that are tagged with the attribute `[BurstCompile]`. Default is checked.
- **Enable Safety Checks**: Has three options in a sub-menu:
  - **Off** - disable safety checks across all Burst jobs/function-pointers. This is a dangerous setting that should only be used when you want more realistic profiling results from in-editor captures. A reload of the Unity Editor will always reset this to On.
  - **On** - Burst enables safety checks on code that uses collection containers (e.g `NativeArray<T>`). Checks include job data dependency and container indexes out of bounds. This is the default.
  - **Force On** - Force safety checks on _even for jobs/function-pointers that have_ `DisableSafetyChecks = true`. Prior to reporting any Burst bug this option should be set to rule out any problems that safety checks could have caught.
- **Synchronous Compilation**: When checked, Burst will compile synchronously - See [`[BurstCompile]` options](AdvancedUsages.md#synchronous-compilation). Default is unchecked.
- **Native Debug Mode Compilation**: When checked, Burst will disable optimizations on all code compiled, in order to make it easier to debug via a native debugger - See [Native Debugging](DebuggingAndProfiling.md#native-debugging). Default is unchecked.
- **Show Timings**: When checked, Burst logs the time it takes to JIT compile a Job in the Editor. Default is unchecked.
- **Open Inspector...**: Opens the [Burst Inspector Window](#burst-inspector).

# Burst Inspector

The Burst Inspector window displays all the Jobs and other Burst compile targets in the project. Open the Inspector from the Jobs menu (**Jobs** > **Burst Inspector**).

The inspector allows you to view all the Jobs that can be compiled, you can also then check the generated intermediate and native assembly code.

![Burst Inspector](../images/burst-inspector.png)

On the left pane of the window, **Compile Targets** provides an alphabetically sorted list of the Jobs in the project that Burst can compile. Note that the disabled Jobs in the list don't have the `[BurstCompile]` attribute.

On the right pane, the window displays options for viewing the assembly and intermediate code for the selected compile target.

**To view the disassembly for a Job:**

1. Select an active compile target from the _left pane_.
3. Switch between the different tabs to display the details:
   * **Assembly** provides the final optimized native code generated by Burst
   * **.NET IL** provides a view on the original .NET IL extracted from the Job method
   * **LLVM IR (Unoptimized)** provides a view on the internal LLVM IR before optimizations.
   * **LLVM IR (Optimized)** provides a view on the internal LLVM IR after optimizations.
   * **LLVM IR Optimization Diagnostics** provides detailed LLVM diagnostics of the optimizations (i.e if they succeeded or failed).
4. You can also turn on different options:
   * There is a dropdown to specify what output to show. If you click the **Copy to Clipboard** button it'll copy the output that matches what you specify here (so if you have coloured output you'll get all the `<color=#444444>foo</color>` tags). In general its best to view the coloured output in the inspector, but copy the plain output if you want to move it into an external tool.
      * **Plain (No debug information)** - raw output.
      * **Plain (With debug information)** - same as above but with debug information included too.
      * **Enhanced (Minimal debug information)** - enhanced output has the line information interweaved with the assembly to guide you as to what line in your code matches what assembly output.
      * **Enhanced (Full debug information)** - same as above but with debug information included too.
      * **Coloured (Minimal debug information)** - same as the enhanced output but the assembly is colourised nicely to aid reading.
      * **Coloured (Full debug information)** - same as above but with debug information included too.
   * The **Safety Checks** option generates code that includes container access safety checks (e.g check if a job is writing to a native container that is readonly).

# Command-line Options

You can pass the following options to the Unity Editor on the command line to control Burst:

- `--burst-disable-compilation` — turns Burst off.
- `--burst-force-sync-compilation` — Burst always compiles synchronously. See [`[BurstCompile]` options](AdvancedUsages.md#synchronous-compilation).

# Just-In-Time (JIT) vs Ahead-Of-Time (AOT) Compilation

When working on your projects in the editor (Play Mode), Burst works in a Just-In-Time (JIT) fashion. Burst will compile your code at the point that it is to be used. By default this is done asnychronously which means your code will be running under the default mono JIT until the compilation by Burst has been completed.
You can control this behaviour via [`[BurstCompile]` options](AdvancedUsages.md#synchronous-compilation).

However, when you build your project into a Standalone Player, Burst will instead compile all the supported code Ahead-Of-Time (AOT). AOT compilation at present, requires access to some linker tools for the respective platforms (similar to the requirements of IL2CPP). See [Burst AOT Requirements](StandalonePlayerSupport.md#burst-aot-requirements).

See [Standalone Player Support](StandalonePlayerSupport.md).